﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;

namespace VA.PPMS.CRM.Plugins.Provider
{
    public class CareSiteCreate : IPlugin
    {
        private ITracingService tracingService;
        private const string PluginName = "CareSiteCreate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_caresite")
                    return;

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    tracingService.Trace("Check for matching care site by composite");
                    // Check for exclusion by demographics
                    Entity match = GetMatchingCareSiteByComposite(service, entity);
                    if (match != null)
                    {
                        tracingService.Trace("Match found, set reference to existing care site");
                        entity.Attributes["ppms_caresite"] = match.ToEntityReference();
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }

        // Retrieve matching provider records based on a composite of LEIE Exclusion fields
        public Entity GetMatchingCareSiteByComposite(IOrganizationService service, Entity entity)
        {
            EntityCollection results = new EntityCollection();
            string addressComposite = entity.GetAttributeValue<string>("ppms_name");
            int accountState = 0; // only active

            //define query
            QueryExpression query = new QueryExpression("ppms_caresite");
            query.ColumnSet.AddColumns("ppms_caresiteid", "ppms_name");

            // add address and state filter
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_name", ConditionOperator.Like, addressComposite);
            filter.AddCondition("statecode", ConditionOperator.Equal, accountState);
            filter.AddCondition("ppms_caresiteid", ConditionOperator.NotEqual, entity.Id);
            query.Criteria.AddFilter(filter);

            // retrieve results, return first match
            var matches = service.RetrieveMultiple(query);
            if (matches != null && matches.Entities != null && matches.Entities.Any())
            {
                return matches.Entities.First();
            }
            else
            {
                return null;
            }
        }
    }
}
